/****************************************************************************
**
** Copyright (C) 2020 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Safe Renderer module
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/
import QtQuick 2.0
import Qt.SafeRenderer 1.1

/*!
    \qmltype SafeText
    \inqmlmodule QtSafeRenderer
    \brief Provides a QML text type which can be rendered in the Qt Safe Renderer runtime.

    The \l {text} can be changed on runtime if \l {runtimeEditable} is set to
    \c{true}.
    The SafeText object communicates the scene position changes to the Safe Renderer runtime.
    For that purpose, a unique \l {objectName} property must be set.

    The following code provides an example of the SafeText type:

    \code
    SafeText {
        id: safeText
        objectName: "safeText"
        x: 256
        y: 8
        text: "Safe text.."
        font.pixelSize: 12
    }
    \endcode
*/

SafeRendererText {
     id: textObject
     width: 128
     height: 64
     font.family: "Times New Roman"
     font.pixelSize: 16
     runtimeEditable: false
     onPositionChanged : {
         SafeMessage.moveItem(textObject.objectName, pos)
     }
     onTextChanged: {
         if (runtimeEditable) {
            SafeMessage.setText(textObject.objectName, text)
         }
     }

     onColorChanged: {
         if (runtimeEditable) {
             SafeMessage.setColor(textObject.objectName, color)
         }
     }
}
